<?php

namespace HulkPlugins\ElementorPro\Forms\GSheetConnector\Controllers;

use Elementor\Controls_Manager;
use ElementorPro\Modules\Forms\Classes\Action_Base;
use Google\Service\Sheets;
use Google\Service\Sheets\BatchUpdateSpreadsheetRequest;
use Google\Service\Sheets\ValueRange;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Models\SheetModel;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Services\FormActionService;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Services\GoogleSheetsService;
use Throwable;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\PREFIX;

class FormActionController extends Action_Base {

	private string $prefix = PREFIX . 'gsheetca';

	public function get_name(): string {
		return 'hulk_epfgsc_gsheet_connector_action';
	}

	public function get_label(): string {
		return esc_attr__( 'Google Sheet Connector', 'gsheet-connector-for-elementor-forms' );
	}

	public function run( $record, $ajax_handler ) {
		$service  = FormActionService::getInstance();
		$settings = $record->get( 'form_settings' );
		$sheetData = sanitize_text_field( $settings[ PREFIX . 'gsheetcasheetName' ] ?? '' );
		$sheetData = explode( ',', $sheetData );
		$spreadsheetId = $sheetData[0] ?? '';
		$sheetId = (int)($sheetData[1] ?? 0);
		if ( ! isset( $spreadsheetId ) ) {
			return;
		}

		$options = [
			'valueInputOption' => 'RAW', // or 'USER_ENTERED' for automatic formatting
		];

		$data = [];

		$submit_actions = $record->get_form_settings( 'submit_actions' );
		$isCollectSubmissions = in_array( 'save-to-database', $submit_actions, true );
		$formHeaders = $record->get_form_settings( $this->prefix . 'formHeaders' );
		$isEntryId = $record->get_form_settings( $this->prefix . 'entryId' ) === 'yes';
		$isEntryDate = $record->get_form_settings( $this->prefix . 'entryDate' ) === 'yes';
		$isPostId = $record->get_form_settings( $this->prefix . 'postId' ) === 'yes';
		$isUserName = $record->get_form_settings( $this->prefix . 'userName' ) === 'yes';
		$isUserIp = $record->get_form_settings( $this->prefix . 'userIp' ) === 'yes';
		$isUserAgent = $record->get_form_settings( $this->prefix . 'userAgent' ) === 'yes';
		$isUserId = $record->get_form_settings( $this->prefix . 'userId' ) === 'yes';
		$isReferrer = $record->get_form_settings( $this->prefix . 'referrer' ) === 'yes';
		$isFreezeHeader = $record->get_form_settings( $this->prefix . 'freezeHeader' ) === 'yes';

		$userId = get_current_user_id();
		$formMeta = $record->get_form_meta( [ 'date','time','page_url','page_title','user_agent','remote_ip' ] );

		// Form Fields
		$fields = $record->get( 'fields' );
		foreach ( $fields as $field ) {
			if ( isset( $field['title'] ) && is_array( $formHeaders ) && in_array( $field['id'], $formHeaders, true ) ) {
				$data[ $field['id'] ]['title'] = sanitize_text_field( $field['title'] );
				$data[ $field['id'] ]['value'] = sanitize_text_field( $field['value'] ?? '' );
			}
		}

		// Entry ID
		if ( $isEntryId === true ) {
			$data['entry-id']['title'] = esc_attr__( 'Entry ID', 'gsheet-connector-for-elementor-forms' );
			$data['entry-id']['value'] = '';
			if ( $isCollectSubmissions === true ) {
				$lastEntry = $service->getSubmissionLastEntry();
				if ( isset( $lastEntry['id'] ) ) {
					$data['entry-id']['value'] = intval( $lastEntry['id'] );
				}
			}
		}

		// Entry Date
		if ( $isEntryDate === true ) {
			$data['entry-date']['title'] = esc_attr__( 'Entry Date', 'gsheet-connector-for-elementor-forms' );
			$data['entry-date']['value'] = "{$formMeta['date']['value']} {$formMeta['time']['value']}";
		}

		// Post ID
		if ( $isPostId === true ) {
			$data['post-id']['title'] = esc_attr__( 'Post ID', 'gsheet-connector-for-elementor-forms' );
			$data['post-id']['value'] = '';
			$form_post_id = $record->get_form_settings( 'form_post_id' );
			if ( ! empty( $form_post_id ) ) {
				$data['post-id']['value'] = intval( $form_post_id );
			}
		}

		// User Name
		if ( $isUserName === true ) {
			$data['user-name']['title'] = esc_attr__( 'User Name', 'gsheet-connector-for-elementor-forms' );
			$data['user-name']['value'] = '';
			$user = get_user_by( 'id', $userId );
			if ( ! empty( $user->display_name ) ) {
				$data['user-name']['value'] = sanitize_text_field( $user->display_name );
			} elseif ( ! empty( $user->user_login ) ) {
				$data['user-name']['value'] = sanitize_text_field( $user->user_login );
			}
		}

		// User IP
		if ( $isUserIp === true ) {
			$data['user-ip']['title'] = esc_attr__( 'User IP', 'gsheet-connector-for-elementor-forms' );
			$data['user-ip']['value'] = '';
			if ( ! empty( $formMeta['remote_ip']['value'] ) ) {
				$data['user-ip']['value'] = sanitize_text_field( $formMeta['remote_ip']['value'] );
			}
		}

		// User Agent
		if ( $isUserAgent === true ) {
			$data['user-agent']['title'] = esc_attr__( 'User Agent', 'gsheet-connector-for-elementor-forms' );
			$data['user-agent']['value'] = '';
			if ( ! empty( $formMeta['user_agent']['value'] ) ) {
				$data['user-agent']['value'] = sanitize_text_field( $formMeta['user_agent']['value'] );
			}
		}

		// User ID
		if ( $isUserId === true ) {
			$data['user-id']['title'] = esc_attr__( 'User ID', 'gsheet-connector-for-elementor-forms' );
			$data['user-id']['value'] = $userId;
		}

		// Referrer
		if ( $isReferrer === true ) {
			$data['referrer']['title'] = esc_attr__( 'Referrer', 'gsheet-connector-for-elementor-forms' );
			$data['referrer']['value'] = '';
			$referer = $service->getReferer( $formMeta );
			if ( ! empty( $referer ) && is_string( $referer ) ) {
				$data['referrer']['value'] = sanitize_url( $referer );
			}
		}

		try {
			// Sheets service
			$googleSheetService = GoogleSheetsService::getInstance();
			$client = $googleSheetService->getClient();
			$sheetsService = new Sheets( $client );
			$spreadsheet = $sheetsService->spreadsheets->get( $spreadsheetId );
			$sheets = $spreadsheet->getSheets();
			foreach ( $sheets as $sheet ) {
				$properties = $sheet->getProperties();
				if ( $properties->getSheetId() === $sheetId ) {
					$sheetName = $properties->getTitle();
					/*---------------------------------------------------------------
					## Header logic
					---------------------------------------------------------------*/
					// Header range
					$headerRange = "$sheetName!1:1";
					// Fetch the current headers from the top row
					$headerResponse = $sheetsService->spreadsheets_values->get( $spreadsheetId, $headerRange );
					$existingHeaders = $headerResponse->values[0] ?? []; // If no values exist, use an empty array

					// Identify missing headers
					$missingHeaders = [];
					foreach ( $data as $item ) {
						if ( ! in_array( $item['title'], $existingHeaders, true ) ) {
							$missingHeaders[] = $item['title'];
						}
					}

					// If there are missing headers, add them
					if ( ! empty( $missingHeaders ) ) {

						// Make header bold
						$requestsHeaderBold = $service->boldHeaderRequest( $sheetId );
						// Create the BatchUpdate request body
						$batchUpdateRequest = new BatchUpdateSpreadsheetRequest( [ 'requests' => $requestsHeaderBold ] );
						// Execute the batchUpdate request
						$sheetsService->spreadsheets->batchUpdate( $spreadsheetId, $batchUpdateRequest );

						// Calculate where to insert the missing headers
						$updatedHeaders = $existingHeaders; // Start with the current headers
						foreach ( $missingHeaders as $header ) {
							// Find the first empty slot or append to the end
							$updatedHeaders[] = $header;
						}
						// Update the entire top row with the combined headers
						$body = new ValueRange( [ 'values' => [ $updatedHeaders ] ] );
						$sheetsService->spreadsheets_values->update( $spreadsheetId, $headerRange, $body, $options );
					}

					// Freeze header
					$currentFrozenRowCount = $properties->getGridProperties()->getFrozenRowCount();
					if ( ( $isFreezeHeader === true ) && ( $currentFrozenRowCount < 1 ) ) {
						$requestsHeaderFreeze = $service->freezeHeaderRequest( $sheetId, 1 );
						$batchUpdateRequest = new BatchUpdateSpreadsheetRequest( [ 'requests' => $requestsHeaderFreeze ] );
						$sheetsService->spreadsheets->batchUpdate( $spreadsheetId, $batchUpdateRequest );
						// Unfreeze header
					} elseif ( ( $isFreezeHeader !== true ) && ( $currentFrozenRowCount >= 1 ) ) {
						$requestsHeaderFreeze = $service->freezeHeaderRequest( $sheetId, 0 );
						$batchUpdateRequest = new BatchUpdateSpreadsheetRequest( [ 'requests' => $requestsHeaderFreeze ] );
						$sheetsService->spreadsheets->batchUpdate( $spreadsheetId, $batchUpdateRequest );
					}

					/*---------------------------------------------------------------
					## Body logic
					---------------------------------------------------------------*/
					// Re-fetch the current headers from the top row
					$headerResponseNew = $sheetsService->spreadsheets_values->get( $spreadsheetId, $headerRange );
					$existingHeadersNew = $headerResponseNew->values[0] ?? []; // If no values exist, use an empty array

					$fullRange = "$sheetName!A1:Z";
                    $response   = $sheetsService->spreadsheets_values->get( $spreadsheetId, $fullRange );
                    $getValues = $response->getValues();
					$row = $getValues ? count( $getValues ) + 1 : 1;
                    $range = "$sheetName!A$row:Z";

					// Prepare a row for the values to append
					$values = array_fill( 0, count( $data ), '' ); // Fill with empty strings
					foreach ( $data as $item ) {
						$index = array_search( $item['title'], $existingHeadersNew, true ); // Find the index of the header
						if ( $index !== false ) {
							$values[ $index ] = $item['value']; // Set the value in the correct column
						}
					}

                    // Prepare the request body
                    $body = new ValueRange( [ 'values' => [ $values ] ] );
                    // Append the row to the sheet
                    $sheetsService->spreadsheets_values->append(
                        $spreadsheetId,
                        $range,
                        $body,
                        $options
                    );
				}
			}
		} catch ( Throwable $throwable ) {
			$ajax_handler->add_error_message( $throwable->getMessage() );
		}
	}

	public function register_settings_section( $form ) {

		$form->start_controls_section(
			$this->prefix . 'section',
			[
				'label'     => esc_attr__( 'Google Sheet Connector', 'gsheet-connector-for-elementor-forms' ),
				'condition' => [
					'submit_actions' => $this->get_name(),
				],
			]
		);

		$form->add_control(
			$this->prefix . 'createNew',
			[
				'label' => '',
				'type'  => Controls_Manager::BUTTON,
				'text'  => esc_html__( 'Create New', 'gsheet-connector-for-elementor-forms' ),
				'event' => 'namespace:editor:hulk_epfgsc_createNew',
			]
		);

		$form->add_control(
			$this->prefix . 'sheetName',
			[
				'label'       => esc_html__( 'Select Sheet Name', 'gsheet-connector-for-elementor-forms' ),
				'type'        => Controls_Manager::SELECT,
				'separator'   => 'before',
				'label_block' => true,
				'options'     => [],
			]
		);

		$form->add_control(
			$this->prefix . 'sheetTabName',
			[
				'label'       => esc_html__( 'Select Sheet Tab Name', 'gsheet-connector-for-elementor-forms' ),
				'type'        => Controls_Manager::SELECT,
				'label_block' => true,
				'options'     => [],
			]
		);

		$form->add_control(
			$this->prefix . 'viewSheet',
			[
				'label'     => '',
				'type'      => Controls_Manager::BUTTON,
				'text'      => esc_html__( 'View Google Sheet', 'gsheet-connector-for-elementor-forms' ),
				'event'     => 'namespace:editor:hulk_epfgsc_sheetView',
				'condition' => [
					$this->prefix . 'sheetTabName!' => '',
				],
			]
		);

		$form->add_control(
			$this->prefix . 'sheetFetch',
			[
				'label' => '',
				'type'  => Controls_Manager::BUTTON,
				'text'  => esc_html__( 'Click here to fetch Google Sheets', 'gsheet-connector-for-elementor-forms' ),
				'event' => 'namespace:editor:hulk_epfgsc_sheetFetch',
			]
		);

		$form->add_control(
			$this->prefix . 'apiError',
			[
				'label'   => '',
				'type'    => Controls_Manager::RAW_HTML,
				'raw'     => '',
				'classes' => 'apiErrorControl',
			]
		);

		$sheetModel = SheetModel::getInstance();
		$isDisplay = empty( $sheetModel->getSheets() ) ? 'block' : 'none';

		$form->add_control(
			$this->prefix . 'authError',
			[
				'label'   => '',
				'type'    => Controls_Manager::RAW_HTML,
				'raw'     => '<p style="display: ' . $isDisplay . '"><strong>' . esc_attr__( 'Authentication Required', 'gsheet-connector-for-elementor-forms' ) . ':</strong> ' . esc_attr__( 'You must have to', 'gsheet-connector-for-elementor-forms' ) . " <a href='" . admin_url( 'admin.php?page=gsheet-connector-for-elementor-forms-settings' ) . "' target='_blank'>" . esc_attr__( 'Authenticate using your Google Account', 'gsheet-connector-for-elementor-forms' ) . '</a> ' . esc_attr__( 'along with Google Drive and Google Sheets Permissions in order to enable the settings for configuration.', 'gsheet-connector-for-elementor-forms' ) . '</p>',
				'classes' => 'authErrorControl',
			]
		);

		$form->add_control(
			$this->prefix . 'formHeaders',
			[
				'label'       => esc_html__( 'Form Fields (Headers)', 'gsheet-connector-for-elementor-forms' ),
				'type'        => Controls_Manager::SELECT2,
				'separator'   => 'before',
				'label_block' => true,
				'multiple'    => true,
				'classes'     => 'formHeaders',
				'options'     => [],
				'default'     => [],
			]
		);

		$form->add_control(
			$this->prefix . 'defaultHeaders',
			[
				'label' => esc_html__( 'Default Form Fields (Headers)', 'gsheet-connector-for-elementor-forms' ),
				'type'  => Controls_Manager::HEADING,
			]
		);

		$form->add_control(
			$this->prefix . 'entryId',
			[
				'label'        => esc_attr__( 'Entry ID', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'entryDate',
			[
				'label'        => esc_html__( 'Entry Date', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'postId',
			[
				'label'        => esc_html__( 'Post ID', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'userName',
			[
				'label'        => esc_html__( 'User Name', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'userIp',
			[
				'label'        => esc_html__( 'User IP', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'userAgent',
			[
				'label'        => esc_html__( 'User Agent', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'userId',
			[
				'label'        => esc_html__( 'User ID', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'referrer',
			[
				'label'        => esc_html__( 'Referrer', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
			]
		);

		$form->add_control(
			$this->prefix . 'freezeHeader',
			[
				'label'        => esc_html__( 'Freeze Header', 'gsheet-connector-for-elementor-forms' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'yes',
				'separator'    => 'before',
			]
		);

		$form->end_controls_section();
	}

	public function on_export( $element ) {
	}
}
