<?php

namespace HulkPlugins\ElementorPro\Forms\GSheetConnector\Controllers;

use HulkPlugins\ElementorPro\Forms\GSheetConnector\Services\GoogleSheetsService;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Services\SettingsService;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Traits\SingletonTrait;
use Throwable;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\DOCUMENTATION;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\GROUP;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\PLUGIN_DIR_PATH;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\PLUGIN_DIR_URL;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\PREFIX;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\SUPPORT;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\VERSION;

class SettingsController {
	use SingletonTrait;

	private string $id = 'elementor_page_gsheet-connector-for-elementor-forms-settings';
	private SettingsService $service;
	private static SettingsController $instance;
	private string $previewAction = PREFIX . 'preview';

	private function __construct( SettingsService $service ) {
        $this->service = $service;
	}

	public static function getInstance( SettingsService $service ): SettingsController {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self( $service );
		}

		return self::$instance;
	}

	public function init_hooks() {
		add_action( 'admin_menu', [ $this,'admin_menu' ], 1000 );
		add_filter( 'elementor/admin-top-bar/is-active', [ $this, 'admin_top_bar' ], 10, 2 );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );
	}

	/**
	 * Add settings page under elementor
	 * @return void
	 */
	public function admin_menu() {
		add_submenu_page(
			'elementor',
			esc_attr__( 'Google Sheet Connector', 'gsheet-connector-for-elementor-forms' ),
			esc_attr__( 'Google Sheet Connector', 'gsheet-connector-for-elementor-forms' ),
			'manage_options',
			'gsheet-connector-for-elementor-forms-settings',
			[ $this, 'render_settings_page' ],
		);
	}

	/**
	 * Hide the elementor admin top bar on FormKit settings page
	 * @param $is
	 * @param $current_screen
	 *
	 * @return false|mixed
	 */
	public function admin_top_bar( $is, $current_screen ) {
		if ( $this->id === $current_screen->id ) {
			return false;
		}
		return $is;
	}

	/**
	 * Enqueues scripts and styles for settings page
	 * @return void
	 */
	public function enqueue_admin_scripts() {

		if ( ! wp_script_is( 'react-jsx-runtime', 'registered' ) ) {
			wp_register_script(
				'react-jsx-runtime',
				PLUGIN_DIR_URL . 'assets/js/react-jsx-runtime.js',
				[ 'react' ],
				'18.3.1',
				true
			);
		}

		$assets = require_once PLUGIN_DIR_PATH . 'build/admin/settings/index.asset.php';

        wp_register_style(
            'gsheet-connector-for-elementor-forms-settings',
            PLUGIN_DIR_URL . 'build/admin/settings/index.css',
            [ 'wp-components' ],
	        $assets['version'],
        );

        wp_register_script(
            'gsheet-connector-for-elementor-forms-settings',
            PLUGIN_DIR_URL . 'build/admin/settings/index.js',
	        $assets['dependencies'],
	        $assets['version'],
            true
        );
	}

	/**
	 * Render settings
	 * @return void
	 */
	public function render_settings_page() {
		echo /** @lang text */ '<div class="hulk-settings" id="hulk-epfgsc-settings"></div>';
		wp_enqueue_script( 'gsheet-connector-for-elementor-forms-settings-runtime' );
		wp_enqueue_style( 'gsheet-connector-for-elementor-forms-settings' );
        wp_enqueue_script( 'gsheet-connector-for-elementor-forms-settings' );

		// Load translation files for javascript
        wp_set_script_translations(
            'gsheet-connector-for-elementor-forms-settings',
            'gsheet-connector-for-elementor-forms',
            PLUGIN_DIR_PATH . 'languages'
        );

		// Get auto code from get request
		$authCode = filter_input( INPUT_GET, 'code', FILTER_SANITIZE_SPECIAL_CHARS );

		// Get saved settings
		$settings = &$this->service->getSettings();

		// Get google sheets service instance
		$googleSheetService = GoogleSheetsService::getInstance();

		// Check (Choose Google API Setting) option value is manual
		$isManual = ! empty( $settings['manual_clientId'] ) && ! empty( $settings['manual_clientSecret'] );

		// Get auth code from url
		if ( $authCode && $isManual ) {
			$settings['manual_clientToken'] = $authCode;
		} else {
			$authCode = '';
		}

		$settings['authCode'] = $authCode;

		// Manual auth
		if ( empty( $authCode ) && $isManual ) {
			try {
				$client = $googleSheetService->getManualClient( $settings['manual_clientId'], $settings['manual_clientSecret'] );
				$tokenData = $googleSheetService->getClientAuth( $client, 'manual' );

				// Auth URL
				if ( isset( $tokenData['authUrl'] ) ) {
					$settings['manual_authUrl'] = $tokenData['authUrl'];
				}

				// Set the token data
				if ( ! empty( $tokenData['access_token'] ) ) {
					$client->setAccessToken( $tokenData );
				}

				// If the access token is not expired then get the user info
				if ( ! $client->isAccessTokenExpired() ) {
					$user = $googleSheetService->getUserInfo( $client, $tokenData );
					$settings['manual_connectUserEmail'] = $user->getEmail();
				}
			} catch ( Throwable $e ) {
				$settings['manual_clientAuthError'] = $e->getMessage();
			}
		}

        wp_localize_script(
            'gsheet-connector-for-elementor-forms-settings',
            PREFIX . 'settings',
            [
				'VERSION'         => VERSION,
				'SUPPORT'         => SUPPORT,
				'PREFIX'          => PREFIX,
				'GROUP'           => GROUP,
				'DOCUMENTATION'   => DOCUMENTATION,
				'PLUGIN_DIR_URL'  => PLUGIN_DIR_URL,
				'restUrl'         => rest_url(),
				'adminColor'      => get_user_option( 'admin_color', get_current_user_id() ),
				'nonce'           => wp_create_nonce( 'wp_rest' ),
				'settings'        => $settings,
				'hostname'        => $this->service->getHostname(),
	            'settingsPageUrl' => admin_url( 'admin.php?page=gsheet-connector-for-elementor-forms-settings' ),
            ]
        );
	}
}
