<?php

namespace HulkPlugins\ElementorPro\Forms\GSheetConnector\Models;

use HulkPlugins\ElementorPro\Forms\GSheetConnector\Traits\SingletonTrait;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\PREFIX;

class SettingsModel {
	use SingletonTrait;

	private array $attributes = [];
	private string $optionName = PREFIX . 'settings';
	private array $defaultOptions = [
		'updatedAt'           => '',
		// Sign manual options
		'manual_clientId'     => '',
		'manual_clientSecret' => '',
		'manual_clientToken'  => '',
		'manual_tokenData'    => [
			'access_token'  => '',
			'expires_in'    => 0,
			'refresh_token' => '',
			'scope'         => '',
			'token_type'    => '',
			'id_token'      => '',
		],
	];

	public static function getInstance(): SettingsModel {
		return self::__getInstance();
	}

	/**
	 * Deletes the settings from the database.
	 * @return bool
	 */
	public function deleteSettings(): bool {
		return delete_option( $this->optionName );
	}

	/**
	 * Sanitized tokenData
	 * @param array $data
	 *
	 * @return array
	 */
	public function sanitizeTokenData( array $data ): array {
		$sanitized = [];
		foreach ( $data as $key => $value ) {
			$key = sanitize_text_field( $key );
			switch ( $key ) {
				case 'expires_in':
					$sanitized[ $key ] = intval( $value );
					break;
				default:
					$sanitized[ $key ] = sanitize_text_field( $value );
					break;
			}
		}

		return $sanitized;
	}

	/**
	 * Sanitizes settings
	 *
	 * @param array $input
	 *
	 * @return array Sanitized associative array
	 */
	public function sanitizeSettings( array $input ): array {
		$sanitized = [];
		foreach ( $input as $key => $value ) {
			$key = sanitize_text_field( $key );
			// Skip non exist option
			if ( ! array_key_exists( $key, $this->defaultOptions ) ) {
				continue;
			}
			switch ( $key ) {
                case 'manual_tokenData':
                    $sanitized[ $key ] = $this->sanitizeTokenData( is_array( $value ) ? $value : [] );
                    break;
				default:
					$sanitized[ $key ] = sanitize_text_field( $value );
					break;
			}
		}
		return $sanitized;
	}

	/**
	 * Updates the settings in the database.
	 *
	 * @param array $settings Associative array with the settings to update.
	 *
	 * @return bool True if the settings were updated, false otherwise.
	 */
	public function updateSettings( array $settings ): bool {
		$settings['updatedAt'] = current_time( 'mysql' );
		$updated = update_option( $this->optionName, $this->sanitizeSettings( $settings ) );
		if ( $updated ) {
			// Clear the cached attributes so the next getSettings fetches fresh data.
			$this->attributes = [];
		}
		return $updated;
	}

	/**
	 * Get all settings
	 * @return array
	 */
	public function &getSettings(): array {
		if ( empty( $this->attributes ) ) {
			$options = get_option( $this->optionName, $this->defaultOptions );
			$attributes = array_merge( $this->defaultOptions, is_array( $options ) ? $options : [] );
			$this->attributes = array_map(
				function ( $item ) {
					if ( is_string( $item ) ) {
						return trim( $item );
					} else {
						return $item;
					}
				},
				$attributes
			);
		}
		return $this->attributes;
	}
}
