<?php

namespace HulkPlugins\ElementorPro\Forms\GSheetConnector\Routes;

use HulkPlugins\ElementorPro\Forms\GSheetConnector\Models\SettingsModel;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Services\GoogleSheetsService;
use Throwable;
use WP_Error;
use WP_HTTP_Response;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\GROUP;

class ClientAuthRoute {

	public function __construct() {

		// Client auth manual route
		register_rest_route(
			GROUP . '/v1',
			'/client-auth-manual',
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'client_auth_manual_callback' ],
				'permission_callback' => fn() => current_user_can( 'manage_options' ),
			]
		);

		// Client auth manual deactivate
		register_rest_route(
			GROUP . '/v1',
			'/client-auth-deactivate',
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'client_auth_deactivate_callback' ],
				'permission_callback' => fn() => current_user_can( 'manage_options' ),
			]
		);
	}

	/**
	 * Client auth manual callback
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function client_auth_manual_callback( WP_REST_Request $request ) {
		$params = $request->get_params();

		// Save the settings to the database
		$settingsModel = SettingsModel::getInstance();
		$settingsModel->updateSettings( $params );

		// Sanitize client ID and Secret
		$clientId = sanitize_text_field( $params['manual_clientId'] ?? '' );
		$clientSecret = sanitize_text_field( $params['manual_clientSecret'] ?? '' );

		if ( empty( $clientId ) ) {
			return new WP_REST_Response(
				esc_attr__( 'Client ID is required.', 'gsheet-connector-for-elementor-forms' ),
				400
			);
		}

		if ( empty( $clientSecret ) ) {
			return new WP_REST_Response(
				esc_attr__( 'Client secret is required.', 'gsheet-connector-for-elementor-forms' ),
				400
			);
		}

		try {
			$googleSheetService = GoogleSheetsService::getInstance();
			$client = $googleSheetService->getManualClient( $clientId, $clientSecret );
			$tokenData = $googleSheetService->getClientAuth( $client, 'manual' );

			// Set the token data
			if ( ! empty( $tokenData['access_token'] ) ) {
				$client->setAccessToken( $tokenData );
			}

			// If the access token is not expired then get the user info
			if ( ! $client->isAccessTokenExpired() ) {
				$user = $googleSheetService->getUserInfo( $client, $tokenData );
				$tokenData['email'] = $user->getEmail();
			}

			// Success message
			$tokenData['message'] = esc_attr__( 'Your Google Access Code is Authorized and Saved.', 'gsheet-connector-for-elementor-forms' );
		} catch ( Throwable $e ) {
			return new WP_REST_Response( $e->getMessage(), 400 );
		}

		return rest_ensure_response( $tokenData );
	}

	/**
	 * Client auth deactivate callback
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function client_auth_deactivate_callback() {

		// Get the settings
		$settingsModel = SettingsModel::getInstance();
		$settings = &$settingsModel->getSettings();

		$resetData = [
			'manual_clientToken' => '',
			'manual_tokenData'   => [],
		];

		// Reset client auth data
		$settingsModel->updateSettings( array_merge( $settings, $resetData ) );

		return rest_ensure_response( esc_attr__( 'Your account is removed. Reauthenticate again to integrate Elementor Form with Google Sheet.', 'gsheet-connector-for-elementor-forms' ) );
	}
}
