<?php

namespace HulkPlugins\ElementorPro\Forms\GSheetConnector\Routes;

use Exception;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Models\SheetModel;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Services\GoogleSheetsService;
use Throwable;
use WP_Error;
use WP_HTTP_Response;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use const HulkPlugins\ElementorPro\Forms\GSheetConnector\GROUP;

class SheetsRoute {

	public function __construct() {

		register_rest_route(
			GROUP . '/v1',
			'/create-spreadsheet',
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'create_spreadsheet_callback' ],
				'permission_callback' => '__return_true',
			]
		);

		register_rest_route(
			GROUP . '/v1',
			'/fetch-spreadsheets',
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'fetch_spreadsheets_callback' ],
				'permission_callback' => '__return_true',
			]
		);
	}

	/**
	 * Create a new spreadsheet
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function create_spreadsheet_callback( WP_REST_Request $request ) {
		$params = $request->get_params();
		if ( ! isset( $params['prompt'] ) ) {
			return new WP_REST_Response(
				esc_attr__( 'Spreadsheet name not found!', 'gsheet-connector-for-elementor-forms' ),
				400
			);
		}
		$name = sanitize_text_field( $params['prompt'] );
		try {
			$googleSheetService = GoogleSheetsService::getInstance();
			$client = $googleSheetService->getClient();
			$sheets = $googleSheetService->createSpreadsheet( $client, $name );
			return rest_ensure_response( $sheets );
		} catch ( Throwable $e ) {
			return new WP_REST_Response( $e->getMessage(), 400 );
		}
	}

	/**
	 * Fetch existing spreadsheets
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 */
	public function fetch_spreadsheets_callback() {
		$sheetModel = SheetModel::getInstance();
		try {
			$googleSheetService = GoogleSheetsService::getInstance();
			$client = $googleSheetService->getClient();
			$sheets = $googleSheetService->getSpreadsheets( $client );
			$sheetModel->updateSheets( $sheets );
			return rest_ensure_response( $sheets );
		} catch ( Throwable $e ) {
			$sheetModel->deleteSheets();
			return new WP_REST_Response( $e->getMessage(), 400 );
		}
	}
}
