<?php

namespace HulkPlugins\ElementorPro\Forms\GSheetConnector\Services;

use Google\Client;
use Google\Service\Drive;
use Google\Service\Exception;
use Google\Service\Oauth2;
use Google\Service\Oauth2\Userinfo;
use Google\Service\Sheets;
use Google\Service\Sheets\Spreadsheet;
use Google_Client;
use Google_Service_Drive;
use Google_Service_Oauth2;
use Google_Service_Sheets;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Models\SettingsModel;
use HulkPlugins\ElementorPro\Forms\GSheetConnector\Traits\SingletonTrait;

class GoogleSheetsService {
	use SingletonTrait;

	private static GoogleSheetsService $instance;
	private ?Google_Client $manualClient = null;
	private ?Google_Client $preAuthClient = null;
	private static ?Google_Client $client = null;

	public static function getInstance(): GoogleSheetsService {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Get client by client id and client secret
	 * @param string $clientId
	 * @param string $clientSecret
	 *
	 * @return Google_Client|null
	 */
	public function getManualClient( string $clientId, string $clientSecret ): ?Google_Client {
		if ( ! $this->manualClient ) {
			$client = new Google_Client();
			$client->setAccessType( 'offline' );
			$client->setPrompt( 'consent' );
			$client->setClientId( $clientId );
			$client->setClientSecret( $clientSecret );
			$client->setScopes(
				[
					Google_Service_Sheets::SPREADSHEETS,
					Google_Service_Drive::DRIVE_METADATA_READONLY,
					Google_Service_Oauth2::USERINFO_EMAIL,
				]
			);
			$redirectUri = wp_get_environment_type() === 'development' ? 'https://localhost' : esc_html( admin_url( 'admin.php?page=gsheet-connector-for-elementor-forms-settings' ) );
			$client->setRedirectUri( $redirectUri );
			$this->manualClient = $client;
		}
		return $this->manualClient;
	}

	/**
	 * Get client token data by client
	 *
	 * @param Client $client
	 * @param string $type: "existing" | "manual"
	 *
	 * @return mixed
	 * @throws Exception
	 */
	public function getClientAuth( Client $client, string $type ) {
		$settingsModel = SettingsModel::getInstance();
		$settings = &$settingsModel->getSettings();

		// Set the access token if available in settings
		if ( ! empty( $settings[ "{$type}_tokenData" ]['access_token'] ) ) {
			$client->setAccessToken( $settings[ "{$type}_tokenData" ] );
		}

		// Check if the access token is expired or about to expire
		if ( $client->isAccessTokenExpired() ) {

			// Try to refresh the access token using the refresh token
			if ( $client->getRefreshToken() ) {

				$tokenData = $client->fetchAccessTokenWithRefreshToken( $client->getRefreshToken() );

				// Handle missing or invalid access token
				if ( empty( $tokenData['access_token'] ) ) {
					$message = esc_attr__( 'Something wrong! Your Auth code may be wrong or expired Please Deactivate and Do Re-Auth Code', 'gsheet-connector-for-elementor-forms' );
					$message .= '<details>';
					$message .= wp_json_encode( $tokenData, JSON_PRETTY_PRINT );
					$message .= '</details>';
					throw new Exception( wp_kses_post( $message ), 400 );
				}

				// Update refresh token if a new one is provided
				$auth = $client->getOAuth2Service();
				$auth->setRefreshToken( $tokenData['refresh_token'] );

				// Save the new token data in settings
				$client->setAccessToken( $tokenData );

				// Save the new token data in settings
				$updated = $settingsModel->updateSettings(
					array_merge( $settings, [ "{$type}_tokenData" => $client->getAccessToken() ] )
				);

				if ( ! $updated ) {
					throw new Exception(
						esc_attr__( 'Unable to update settings while refreshing token.', 'gsheet-connector-for-elementor-forms' ),
						400
					);
				}

				// If no refresh token, try using the authorization code
			} elseif ( ! empty( $settings[ "{$type}_clientToken" ] ) ) {

				$tokenData = $client->fetchAccessTokenWithAuthCode( $settings[ "{$type}_clientToken" ] );

				// Handle missing or invalid access token
				if ( ! empty( $tokenData['access_token'] ) ) {

					// Save the new token data in settings
					$client->setAccessToken( $tokenData );

					// Save the new token data in settings
					$updated = $settingsModel->updateSettings(
						array_merge( $settings, [ "{$type}_tokenData" => $client->getAccessToken() ] )
					);

					if ( ! $updated ) {
						throw new Exception(
							esc_attr__( 'Unable to update settings while fetching access token.', 'gsheet-connector-for-elementor-forms' ),
							400
						);
					}
				} else {
					$message = esc_attr__( 'Something wrong! Your Auth code may be wrong or expired Please Deactivate and Do Re-Auth Code', 'gsheet-connector-for-elementor-forms' );
					$message .= '<details>';
					$message .= wp_json_encode( $tokenData, JSON_PRETTY_PRINT );
					$message .= '</details>';
					throw new Exception( wp_kses_post( $message ), 400 );
				}
			} else {
				// Create a new auth url
                $authUrl = $client->createAuthUrl();
                $settings[ "{$type}_tokenData" ]['authUrl'] = $authUrl;
			}
		}

		// Get fresh settings
		$settings = $settingsModel->getSettings();
		return $settings[ "{$type}_tokenData" ];
	}

	/**
	 * Get user info by client
	 * @param Client $client
	 * @param $token
	 *
	 * @return Userinfo
	 * @throws Exception
	 */
	public function getUserInfo( Client $client, $token ): Userinfo {
		$client->setAccessToken( $token );
		$service = new Oauth2( $client );

		return $service->userinfo->get();
	}

	/**
	 * @return Google_Client|null
	 * @throws Exception
	 */
	public function getClient(): ?Google_Client {
		if ( ! self::$client ) {
			$settingsModel = SettingsModel::getInstance();

			// Get saved settings
			$settings = $settingsModel->getSettings();

			// Get google sheets service instance
			$googleSheetService = self::getInstance();

			if ( ! empty( $settings['manual_clientId'] ) && ! empty( $settings['manual_clientSecret'] ) ) {
				$client = $googleSheetService->getManualClient( $settings['manual_clientId'], $settings['manual_clientSecret'] );
				$tokenData = $googleSheetService->getClientAuth( $client, 'manual' );
			} else {
				throw new Exception(
					esc_attr__( 'Client ID and Client Secret is required!', 'gsheet-connector-for-elementor-forms' ),
					400
				);
			}

			// Set the token data
			if ( empty( $tokenData['access_token'] ) ) {
				throw new Exception(
					esc_attr__( 'Google Sheet Connector Access token is missing.', 'gsheet-connector-for-elementor-forms' ),
					400
				);
			}

			$client->setAccessToken( $tokenData );

			// If the access token is not expired then get the user info
			if ( $client->isAccessTokenExpired() ) {
				throw new Exception(
					esc_attr__( 'Google Sheet Connector Access token expired.', 'gsheet-connector-for-elementor-forms' ),
					400
				);
			}

			self::$client = $client;
		}

		return self::$client;
	}

	/**
	 * @param Client $client
	 *
	 * @return array
	 * @throws Exception
	 */
	public function getSpreadsheets( Client $client ): array {
		$data = [];

		$drive = new Drive( $client );
		$sheets = new Sheets( $client );

		$optParams = [
			'pageSize' => 100, // Number of files to retrieve
			'fields'   => 'nextPageToken, files(id, name)', // Fields to retrieve
			'q'        => "mimeType = 'application/vnd.google-apps.spreadsheet' and trashed = false",
		];

		$results = $drive->files->listFiles( $optParams );

		foreach ( $results->getFiles() as $file ) {
			$options = [];
			$tabs = $sheets->spreadsheets->get( $file->getId() );
			foreach ( $tabs->getSheets() as $sheet ) {
				$properties = $sheet->getProperties();
				$options[ $properties->getSheetId() ] = $properties->getTitle();
			}

			$data[ $file->getId() ] = [
				'title'   => $file->getName(),
				'options' => $options,
			];
		}

		return $data;
	}

	/**
	 * @throws Exception
	 */
	public function createSpreadsheet( Client $client, string $title ): Spreadsheet {
		$service = new Sheets( $client );
		$spreadsheet = new Spreadsheet(
            [
				'properties' => [
					'title' => $title,
				],
            ]
        );

		return $service->spreadsheets->create( $spreadsheet );
	}
}
