import React, {useMemo, useState} from 'react';
import MainHeading from "../../../components/MainHeading";
import {__} from "@wordpress/i18n";
import {Table, Td, Th, Tr} from "../../../components/Table";
import Label from "../../../components/Label";
import {Button, Notice, TextControl} from "@wordpress/components";
import useSettingsStore from "../stores/useSettingsStore";
import useNoticeStore from "../stores/useNoticeStore";
import delay from "../../../utils/delay";
import clientAuthDeactivateAjax from "../utils/clientAuthDeactivateAjax";
import settingsSaveAjax from "../utils/settingsSaveAjax";
import redirectToSettingsPage from "../utils/redirectToSettingsPage";
import clientAuthManualAjax from "../utils/clientAuthManualAjax";
import {DEACTIVATE_CONFIRM_MESSAGE} from "../utils/constants";
import localizeSettings from "../types/localizeSettings";
import copy from "copy-to-clipboard";
import {copySmall} from "@wordpress/icons";

const IntegrationManual = () => {

	const setSuccess = useNoticeStore(state => state.setSuccess);
	const clientId = useSettingsStore(state => state.manual_clientId);
	const clientSecret = useSettingsStore(state => state.manual_clientSecret);
	const clientToken = useSettingsStore(state => state.manual_clientToken);
	const authUrl = useSettingsStore(state => state.manual_authUrl);
	const connectUserEmail = useSettingsStore(state => state.manual_connectUserEmail);
	const error = useSettingsStore(state => state.manual_clientAuthError);
	const authCode = useSettingsStore(state => state.authCode);

	const [saveLoading, setSaveLoading] = useState(false);

	async function handleReset(e: React.MouseEvent<HTMLButtonElement>) {
		e.preventDefault();
		if (saveLoading) {
			return;
		}
		useSettingsStore.setState({
			authCode: '',
			manual_clientId: '',
			manual_clientSecret: '',
			manual_clientToken: '',
			manual_authUrl: '',
			manual_connectUserEmail: '',
			manual_clientAuthError: '',
		});
		redirectToSettingsPage();
		setSaveLoading(true);
		try {
			await delay(200);
			await settingsSaveAjax();
		} catch (err) {
			if (typeof err?.message === "string") {
				useSettingsStore.setState({manual_clientAuthError: err.message});
			} else if (typeof err === 'string') {
				useSettingsStore.setState({manual_clientAuthError: err});
			} else {
				useSettingsStore.setState({manual_clientAuthError: JSON.stringify(err)});
			}
		} finally {
			setSaveLoading(false);
		}
	}

	async function handleSave(e: React.MouseEvent<HTMLButtonElement>) {
		e.preventDefault();
		useSettingsStore.setState({manual_clientAuthError: ''});
		if (saveLoading) {
			return;
		}
		setSaveLoading(true);
		try {
			redirectToSettingsPage();
			useSettingsStore.setState({authCode: ''});
			await delay(200);
			// Auth manual
			const res: any = await clientAuthManualAjax();
			if (res?.authUrl) {
				useSettingsStore.setState({manual_authUrl: res.authUrl});
				setSuccess(res.message);
			}
			if (res?.email) {
				useSettingsStore.setState({manual_connectUserEmail: res.email});
			}
		} catch (err) {
			if (typeof err?.message === "string") {
				useSettingsStore.setState({manual_clientAuthError: err.message});
			} else if (typeof err === 'string') {
				useSettingsStore.setState({manual_clientAuthError: err});
			} else {
				useSettingsStore.setState({manual_clientAuthError: JSON.stringify(err)});
			}
		} finally {
			setSaveLoading(false);
		}
	}

	async function handleDeactivate(e: React.MouseEvent<HTMLButtonElement>) {
		e.preventDefault();
		if (saveLoading) {
			return;
		}
		// Valid user connected email
		if (connectUserEmail) {
			// eslint-disable-next-line no-alert
			if (!confirm(DEACTIVATE_CONFIRM_MESSAGE)) {
				return;
			}
		}
		setSaveLoading(true);
		try {
			const res: any = await clientAuthDeactivateAjax();
			useSettingsStore.setState({
				authCode: '',
				manual_clientToken: '',
				manual_tokenData: {},
				manual_authUrl: '',
				manual_connectUserEmail: '',
				manual_clientAuthError: '',
			});
			redirectToSettingsPage();
			await delay(200);
			const resAuth: any = await clientAuthManualAjax();
			if (resAuth?.authUrl) {
				useSettingsStore.setState({manual_authUrl: resAuth.authUrl});
			}
			if (resAuth?.email) {
				useSettingsStore.setState({manual_connectUserEmail: resAuth.email});
			}
			setSuccess(res);
		} catch (err) {
			if (typeof err?.message === "string") {
				useSettingsStore.setState({manual_clientAuthError: err.message});
			} else if (typeof err === 'string') {
				useSettingsStore.setState({manual_clientAuthError: err});
			} else {
				useSettingsStore.setState({manual_clientAuthError: JSON.stringify(err)});
			}
		} finally {
			setSaveLoading(false);
		}
	}

	const isActivated = useMemo(() => {
		return !!(clientToken?.trim() && !authCode?.trim());
	}, [authCode, clientToken]);

	return (
		<>
			<MainHeading className={'mb-4'}>
				{__('Elementor - Google Sheet Integration')}
			</MainHeading>
			{(!(isActivated || connectUserEmail)) && (
				<Notice
					isDismissible={false}
					status={'warning'}
					className={'mb-4'}
				>
					<h3 className={'mt-0'}>{__('To authenticate with your Google account, follow these steps:')}</h3>
					<ol className={'font-medium mb-0'}>
						<li>{__('First you will need to add Client ID and Client Secret and click save')}</li>
						<li>{__('After that you will need to click "Click here to generate an Authentication Token" button to retrieve your code from Google')}</li>
						<li>{__('Grant permissions for the following:')}</li>
						<ol>
							<li>{__('Google Drive')}</li>
							<li>
								{__('Google Sheets')}
								<div
									className={'text-red-700'}>{__('* Ensure that you enable the checkbox for each of these services.')}</div>
							</li>
						</ol>
						<li>{__('After that you will need to click "Save & Authenticate" button')}</li>
						<li>{__('This will allow the integration to access your Google Drive and Google Sheets')}</li>
					</ol>
				</Notice>
			)}
			<Table>
				<Tr>
					<Th className={'pb-0'}>
						<Label>{__('Client Id')}</Label>
					</Th>
					<Td className={'pb-0'}>
						<TextControl
							autoComplete={'given-name'}
							name={'client-id'}
							__nextHasNoMarginBottom
							className={'w-full max-w-[600px]'}
							readOnly={isActivated || !!authUrl || !!authCode || saveLoading || !!connectUserEmail}
							value={clientId}
							onChange={(value) => {
								useSettingsStore.setState({
									manual_clientId: value,
									manual_authUrl: ''
								});
							}}
						/>
					</Td>
				</Tr>
				<Tr>
					<Th className={'pb-0'}>
						<Label>{__('Client Secret')}</Label>
					</Th>
					<Td className={'pb-0'}>
						<TextControl
							autoComplete={'given-name'}
							name={'client-secret'}
							__nextHasNoMarginBottom
							readOnly={isActivated || !!authUrl || !!authCode || saveLoading || !!connectUserEmail}
							className={'w-full max-w-[600px]'}
							value={clientSecret}
							onChange={(value) => {
								useSettingsStore.setState({
									manual_clientSecret: value,
									manual_authUrl: ''
								});
							}}
						/>
					</Td>
				</Tr>
				{connectUserEmail && (
					<Tr>
						<Th className={'pb-0'}>
							<Label>{__('Connected Email Account')}</Label>
						</Th>
						<Td className={'pb-0'}>
							<p
								className={'m-0 text-white p-1 max-w-[600px] text-center font-bold text-[14px] rounded bg-green-600'}
							>{connectUserEmail}</p>
						</Td>
					</Tr>
				)}
				{(clientId && clientSecret) && (authUrl || clientToken) && (
					<Tr>
						<Th className={'pb-0'}>
							<Label>{__('Client Token')}</Label>
						</Th>
						<Td className={'pb-0'}>
							{authUrl && (
								<a
									href={authUrl}
									target='_blank'
									rel='noopener noreferrer'
									className={'components-button is-secondary is-compact has-text'}
								>
									{__('Click here to generate an Authentication Token')}
								</a>
							) || (clientToken && (
								<TextControl
									className={'w-full max-w-[600px]'}
									__nextHasNoMarginBottom
									value={clientToken}
									readOnly={true}
									onChange={(value) => {
										useSettingsStore.setState({manual_clientToken: value});
									}}
								/>
							))}
						</Td>
					</Tr>
				)}
				<Tr>
					<Th></Th>
					<Td>
						{error && (
							<Notice
								status={'error'}
								__unstableHTML={true}
								isDismissible={false}
								className={'mb-4 max-w-[620px] text-black text-[13px] custom-notice'}
							>{error}</Notice>
						)}
						<div className={'flex gap-4'}>
							<Button
								type="button"
								variant={'primary'}
								onClick={handleSave}
								className={'min-w-24 justify-center'}
								isBusy={saveLoading}
								disabled={saveLoading || isActivated || !!connectUserEmail}
							>
								{authCode ? __('Save & Authenticate') : __('Save')}
							</Button>
							{(isActivated || connectUserEmail) ? (
								<Button
									type="button"
									variant={'secondary'}
									onClick={handleDeactivate}
									className={'min-w-24 justify-center'}
									isBusy={saveLoading}
									disabled={saveLoading}
									title={'On deactivation, all your data saved with authentication will be removed and you need to reauthenticate with your google account.'}
								>
									{__('Deactivate')}
								</Button>
							) : (
								<Button
									type="button"
									variant={'secondary'}
									onClick={handleReset}
									className={'min-w-24 justify-center'}
									isBusy={saveLoading}
									disabled={saveLoading}
								>
									{__('Reset')}
								</Button>
							)}
						</div>
					</Td>
				</Tr>
				<Tr>
					<Th className={'pb-0'}>
						<Label>{__('Authorized Domain')}</Label>
					</Th>
					<Td className={'pb-0'}>
						<div className="flex flex-wrap gap-3">
							<code className={'p-1'}>{localizeSettings.hostname}</code>
							<Button
								type={'button'}
								size={'small'}
								variant={'secondary'}
								onClick={(e: React.MouseEvent<HTMLButtonElement>) => {
									e.preventDefault();
									copy(localizeSettings.hostname);
									setSuccess(__('Copied'));
								}}
								title={__('Copy')}
								icon={copySmall}
							/>
						</div>
					</Td>
				</Tr>
				<Tr>
					<Th className={'pt-1'}>
						<Label>{__('Authorized redirect URI')}</Label>
					</Th>
					<Td className={'pt-1'}>
						<div className="flex flex-wrap gap-3">
							<code className={'p-1'}>{localizeSettings.settingsPageUrl}</code>
							<Button
								type={'button'}
								size={'small'}
								variant={'secondary'}
								onClick={(e: React.MouseEvent<HTMLButtonElement>) => {
									e.preventDefault();
									copy(localizeSettings.settingsPageUrl);
									setSuccess(__('Copied'));
								}}
								title={__('Copy')}
								icon={copySmall}
							/>
						</div>
					</Td>
				</Tr>
			</Table>
		</>
	);
};

export default IntegrationManual;
