import React, {memo, useEffect, useRef} from 'react';
import {SnackbarList} from "@wordpress/components";
import useNoticeStore from "../stores/useNoticeStore";

const timer = 3000 // Adjust the time as needed

const Notice = () => {

	const notices = useNoticeStore(state => state.notices);
	const removeNotice = useNoticeStore(state => state.removeNotice);
	const removeLastItem = useNoticeStore(state => state.removeLastItem);
	const timerRef = useRef<NodeJS.Timeout | string | number | undefined>(0);

	// Start the timer when a new item is added
	useEffect(() => {
		if (notices.length > 0) {
			timerRef.current = setTimeout(() => {
				removeLastItem();
			}, timer);
		}
		// Clear timer when the component unmounts or when the item is removed
		return () => {
			clearTimeout(timerRef.current);
		};
	}, [notices, removeLastItem]);

	// Stop times on mouseenter and restart times on mouseleave
	useEffect(() => {

		function handleMouseEnter() {
			clearTimeout(timerRef.current);
		}

		function handleMouseLeave() {
			timerRef.current = setTimeout(() => {
				removeLastItem();
			}, timer);
		}

		notices.forEach(element => {
			const itemElement = element.ref.current as HTMLDivElement;
			if (itemElement) {
				itemElement.addEventListener('mouseenter', handleMouseEnter, false);
				itemElement.addEventListener('mouseleave', handleMouseLeave, false);
			}
		});

		return () => {
			notices.forEach(element => {
				const itemElement = element.ref.current as HTMLDivElement;
				if (itemElement) {
					itemElement.removeEventListener('mouseenter', handleMouseEnter, false);
					itemElement.removeEventListener('mouseleave', handleMouseLeave, false);
				}
			});
		}
	}, [notices, removeLastItem])

	if (notices.length > 0) {
		return (
			<SnackbarList
				className={'hulk-notice-alert'}
				notices={notices}
				onRemove={removeNotice}
			/>
		);
	}

	return null;
};

export default memo(Notice);
