<?php
/**
 * News filter controls.
 *
 * A control for displaying news posts filters.
 *
 * @since 1.0.0
 */
class Elementor_XElNews_Control_Filter extends \Elementor\Base_Data_Control {

	/**
	 * Get control type.
	 *
	 * Retrieve the control type.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Control type.
	 */
	public function get_type() {
		return 'xelnews_filter';
	}

	/**
	 * Enqueue filters area control scripts.
	 *
	 * Used to register and enqueue custom scripts used by the filters
	 * area control.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function enqueue() {
		wp_register_style( 'xel-news-filter-style', XElNews()->plugin_url() . '/assets/css/styles-admin' . ( is_rtl() ? '-rtl' : '' ) . '.css', array(), XElNews()->version() );
		wp_enqueue_style( 'xel-news-filter-style' );

		wp_register_script( 'xel-news-filter-control', XElNews()->plugin_url() . '/assets/js/news-filter-control.js', array( 'elementor-editor', 'jquery' ), XElNews()->version(), true );
		wp_enqueue_script( 'xel-news-filter-control' );

		wp_localize_script(
			'xel-news-filter-control',
			'xelnews',
			array(
				'ajax'       => admin_url( 'admin-ajax.php' ),
				'nonce'      => wp_create_nonce( 'xelnews-nonce' ),
				'taxonomies' => array(),
			)
		);
	}

	/**
	 * Get filters.
	 *
	 * Retrieve all the available filters.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 * @return array Available filters.
	 */
	public static function get_filters() {
		$post_types = array();
		$taxonomies = array(
			''       => esc_html__( 'Select', 'xel-news' ),
			'author' => 'author',
		);

		$args       = array(
			'publicly_queryable' => true,
		);
		$post_types = get_post_types( $args, 'objects' );

		foreach ( $post_types as $post_type ) {
			$obj_taxonomies = get_object_taxonomies( $post_type->name );

			foreach ( $obj_taxonomies as $taxonomy ) {
				$taxonomies[ $taxonomy ] = $taxonomy;
			}
		}

		return apply_filters(
			'xelnews_control_taxonomies',
			$taxonomies
		);
	}

	/**
	 * Get filters control default settings.
	 *
	 * Retrieve the default settings of the filters control. Used to return
	 * the default settings while initializing the filters control.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return array Filters control default settings.
	 */
	protected function get_default_settings() {
		return array(
			'filters' => self::get_filters(),
		);
	}

	/**
	 * Get filters control default value.
	 *
	 * Retrieve the default value of the filters control. Used to return the
	 * default value while initializing the control.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Filters control default value.
	 */
	public function get_default_value() {
		return '';
	}

	/**
	 * Render filters control output in the editor.
	 *
	 * Used to generate the control HTML in the editor using Underscore JS
	 * template. The variables for the class are available using `data` JS
	 * object.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function content_template() {
		$control_uid = $this->get_control_uid();
		?>
		<div class="elementor-control-field">

			<# if ( data.label ) {#>
				<label for="<?php echo esc_attr( $control_uid ); ?>" class="elementor-control-title">{{{ data.label }}}</label>
			<# } #>

			<div class="elementor-control-input-wrapper">
				<input type="hidden" data-setting="{{ data.name }}" />
			</div>

		</div>

		<# if ( data.description ) { #>
		<div class="elementor-control-field-description">{{{ data.description }}}</div>
		<# } #>
		<?php
	}

}
