<?php

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @link       https://themeatelier.net
 * @since      1.0.0
 *
 * @package     chat-whatsapp-pro
 * @subpackage  chat-whatsapp-pro/src/Frontend
 * @author      ThemeAtelier<themeatelierbd@gmail.com>
 */

namespace ThemeAtelier\ChatWhatsappPro\Frontend;

use ThemeAtelier\ChatWhatsappPro\Frontend\Templates\SingleTemplate;
use ThemeAtelier\ChatWhatsappPro\Frontend\Templates\MultiTemplate;

/**
 * The Frontend class to manage all public facing stuffs.
 *
 * @since 1.0.0
 */
class Frontend
{
    /**
     * The slug of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_slug   The slug of this plugin.
     */
    private $plugin_slug;

    /**
     * The min of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $min   The slug of this plugin.
     */
    private $min;
    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string $plugin_name       The name of the plugin.
     * @param      string $version    The version of this plugin.
     */
    public function __construct()
    {
        $this->min   = defined('WP_DEBUG') && WP_DEBUG ? '' : '.min';
        add_action('wp_footer', [$this, 'chat_whatsapp_pro_content']);
        add_action('wp_ajax_handle_form_submission', [$this, 'handle_form_submission']);
        add_action('wp_ajax_nopriv_handle_form_submission', [$this, 'handle_form_submission']);
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public static function enqueue_scripts()
    {
        $options = get_option('cwp_option');
        $waCustomCss = isset($options['whatsapp-custom-css']) ? $options['whatsapp-custom-css'] : '';
        $waCustomJs = isset($options['whatsapp-custom-js']) ? $options['whatsapp-custom-js'] : '';
        $auto_show_popup = isset($options['autoshow-popup']) ? $options['autoshow-popup'] : '';
        $auto_open_popup_timeout = isset($options['auto_open_popup_timeout']) ? $options['auto_open_popup_timeout'] : 0;

        wp_enqueue_style('ico-font');
        wp_enqueue_style('chat-whatsapp-pro-style');

        $custom_css = '';
        include 'dynamic-css/dynamic-css.php';

        if ($waCustomCss) {
            $custom_css .= $waCustomCss;
        }
        wp_add_inline_style('chat-whatsapp-pro-style', $custom_css);
        wp_enqueue_script('moment', array('jquery'), '1.0', true);
        wp_enqueue_script('moment-timezone-with-data');
        wp_enqueue_script('chat-whatsapp-pro-script');

        $frontend_scripts = array(
            'autoShowPopup'         => $auto_show_popup,
            'autoOpenPopupTimeout'     => $auto_open_popup_timeout,
        );

        wp_localize_script('chat-whatsapp-pro-script', 'whatshelp_pro_frontend_script', $frontend_scripts);

        if (!empty($waCustomJs)) {
            wp_add_inline_script('chat-whatsapp-pro-script', $waCustomJs);
        }

        wp_localize_script(
            'chat-whatsapp-pro-script',
            'frontend_scripts',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('chat_whatsapp_nonce'),
            )
        );
    }

    public function chat_whatsapp_pro_content()
    {
        $options = get_option('cwp_option');
        $bubble_include_page = isset($options['bubble_include_page']) ? $options['bubble_include_page'] : '';
        $bubble_exclude_page = isset($options['bubble_exclude_page']) ? $options['bubble_exclude_page'] : '';
        $bubble_button_tooltip = isset($options['bubble_button_tooltip']) ? $options['bubble_button_tooltip'] : '';
        $bubble_button_tooltip_text = isset($options['bubble_button_tooltip_text']) ? $options['bubble_button_tooltip_text'] : '';
        $whatsapp_message_template = isset($options['whatsapp_message_template']) ? $options['whatsapp_message_template'] : '';
        $whatsapp_number = isset($options['opt-number']) ? $options['opt-number'] : '';
        $circle_button_icon_1 = isset($options['circle-button-icon-1']) ? $options['circle-button-icon-1'] : '';
        $circle_button_close_1 = isset($options['circle-button-close-1']) ? $options['circle-button-close-1'] : '';

        $random         = wp_rand(1, 13);
        $bubble_type     = null;
        $buttonLabel    = isset($options['bubble-text']) ? $options['bubble-text'] : '';
        $disableButton_icon = isset($options['disable-button-icon']) ? $options['disable-button-icon'] : '';
        $icons = '';
        if ($disableButton_icon) {
            $icons = '<div class="bubble__icon bubble-animation' . esc_attr($options['circle-animation']) . '">
      <span class="bubble__icon--open"><i class="' . esc_attr($options['circle-button-icon']) . '"></i></span>
      <span class="bubble__icon--close"><i class="' . esc_attr($options['circle-button-close']) . '"></i></span>
      </div>';
        }

        if ($options['opt-button-style'] === '1') {
            $bubble_type = '<button class="wHelp-bubble circle-bubble circle-animation-' . esc_attr($options['circle-animation']) . ' ">
      <span class="open-icon"><i class="' . esc_attr($circle_button_icon_1) . '"></i></span>
      <span class="close-icon"><i class="' . esc_attr($circle_button_close_1) . '"></i></span>';
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '2') {
            $bubble_type = '<button class="wHelp-bubble bubble wHelp-btn-bg">' . wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '3') {
            $bubble_type = '<button class="wHelp-bubble bubble">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '4') {
            $bubble_type = '<button class="wHelp-bubble bubble wHelp-btn-rounded wHelp-btn-bg">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '5') {
            $bubble_type = '<button class="wHelp-bubble bubble wHelp-btn-rounded">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '6') {
            $bubble_type = '<button class="wHelp-bubble bubble wHelp-btn-bg bubble-transparent btn-with-padding">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '7') {
            $bubble_type = '<button class="wHelp-bubble bubble  bubble-transparent btn-with-padding">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '8') {
            $bubble_type = '<button class="wHelp-bubble bubble wHelp-btn-bg wHelp-btn-rounded bubble-transparent btn-with-padding">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        } elseif ($options['opt-button-style'] === '9') {
            $bubble_type = '<button class="wHelp-bubble bubble wHelp-btn-rounded bubble-transparent btn-with-padding">'. wp_kses_post($icons) . esc_attr($buttonLabel);
            if ($bubble_button_tooltip && $bubble_button_tooltip_text) {
                $bubble_type .= '<span class="tooltip_text">' . wp_kses_post($bubble_button_tooltip_text) . '</span>';
            }
            $bubble_type .= '</button>';
        }

        if ($bubble_include_page) {
            if (is_page($bubble_include_page)) {
                if ($options['opt-chat-type'] === 'single') :
                    SingleTemplate::singleTemplate($options, $bubble_type, $random, $whatsapp_message_template, $whatsapp_number);
                else :
                    MultiTemplate::multiTemplate($options, $bubble_type, $random);
                endif;
            }
        } elseif ($bubble_exclude_page) {
            if (!is_page($bubble_exclude_page)) {
                if ($options['opt-chat-type'] === 'single') :
                    SingleTemplate::singleTemplate($options, $bubble_type, $random, $whatsapp_message_template, $whatsapp_number);
                else :
                    MultiTemplate::multiTemplate($options, $bubble_type, $random);
                endif;
            }
        } else {
            if ($options['opt-chat-type'] === 'single') :
                SingleTemplate::singleTemplate($options, $bubble_type, $random, $whatsapp_message_template, $whatsapp_number);
            else :
                MultiTemplate::multiTemplate($options, $bubble_type, $random);
            endif;
        }
    }

    public function handle_form_submission()
    {

        // Verify the nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'chat_whatsapp_nonce')) {
            wp_send_json_error('Invalid nonce');
            wp_die();
        }

        // Sanitize and retrieve POST data
        $name = sanitize_text_field($_POST['name']);
        $message = sanitize_textarea_field($_POST['message']);
        $whatsapp_template = sanitize_text_field($_POST['template']);
        $whatsapp_number  = sanitize_text_field($_POST['agent']);

        // Prepare variables for the template
        $date = date('F j, Y, H:i (h:i A) (\G\M\T O)');
        $siteURL = get_site_url();
        $variables = array('{name}', '{message}', '{date}',  '{siteURL}');
        $values = array($name, $message, $date, $siteURL);
        $text = trim(str_replace($variables, $values, $whatsapp_template));
        $whatsAppURL = 'https://wa.me/' . esc_attr($whatsapp_number) . '?text=' . urlencode($text);

        // Send the WhatsApp URL back to the client
        wp_send_json_success(array('whatsAppURL' => $whatsAppURL));

        wp_die(); // Terminate immediately and return a proper response
    }
}
