<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Adina Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Adina_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}


	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );

        // Register widget scripts
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);

		// Register Editor scripts
		add_action( 'elementor/editor/after_enqueue_scripts', [$this, 'editor_scripts'], 100 );

        // category register
		add_action( 'elementor/elements/categories_registered',[ $this, 'adina_elementor_widget_categories' ] );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'adina' ),
			'<strong>' . esc_html__( 'Adina Core', 'adina' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'adina' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'adina' ),
			'<strong>' . esc_html__( 'Adina Core', 'adina' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'adina' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'adina' ),
			'<strong>' . esc_html__( 'Adina Core', 'adina' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'adina' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		$widgets_manager = \Elementor\Plugin::instance()->widgets_manager;


		// All Dynamic Widgets
		require_once( ADINA_ADDONS . '/dynamic widgets/feature-image.php' );
		require_once( ADINA_ADDONS . '/dynamic widgets/heading.php' );
		require_once( ADINA_ADDONS . '/dynamic widgets/post-navigation.php' );

		
		// Header Elements
		require_once( ADINA_ADDONS . '/header/site-logo.php' );
		

		// Custom post
		require_once( ADINA_ADDONS . '/widgets/service.php' );
		require_once( ADINA_ADDONS . '/widgets/team-member.php' );
		require_once( ADINA_ADDONS . '/widgets/project-slider.php' );
		require_once( ADINA_ADDONS . '/widgets/blog-post.php' );
		require_once( ADINA_ADDONS . '/widgets/service-section.php' );
		require_once( ADINA_ADDONS . '/widgets/project.php' );
		require_once( ADINA_ADDONS . '/widgets/blog-slider.php' );
		require_once( ADINA_ADDONS . '/widgets/project-section.php' );
		require_once( ADINA_ADDONS . '/widgets/project-two.php' );
		



		// Common Widget
		require_once( ADINA_ADDONS . '/widgets/header.php' );
		require_once( ADINA_ADDONS . '/widgets/button.php' );
		require_once( ADINA_ADDONS . '/widgets/animation-title.php' );
		require_once( ADINA_ADDONS . '/widgets/slider-title.php' );
		require_once( ADINA_ADDONS . '/widgets/section-title.php' );
		require_once( ADINA_ADDONS . '/widgets/video-box.php' );
		require_once( ADINA_ADDONS . '/widgets/award.php' );
		require_once( ADINA_ADDONS . '/widgets/testimonial-slider.php' );
		require_once( ADINA_ADDONS . '/widgets/logo-slider.php' );
		require_once( ADINA_ADDONS . '/widgets/counter-logo.php' );
		require_once( ADINA_ADDONS . '/widgets/hero-navbar.php' );
		require_once( ADINA_ADDONS . '/widgets/image-box.php' );
		require_once( ADINA_ADDONS . '/widgets/shape.php' );
		require_once( ADINA_ADDONS . '/widgets/progress-bar.php' );
		require_once( ADINA_ADDONS . '/widgets/animation-shape.php' );
		require_once( ADINA_ADDONS . '/widgets/brand-logo.php' );
		require_once( ADINA_ADDONS . '/widgets/testimonial-slider-section.php' );
		require_once( ADINA_ADDONS . '/widgets/iconbox.php' );
		require_once( ADINA_ADDONS . '/widgets/accordion.php' );
		require_once( ADINA_ADDONS . '/widgets/testimonial-slider-two.php' );
		require_once( ADINA_ADDONS . '/widgets/experience.php' );
		require_once( ADINA_ADDONS . '/widgets/business-category.php' );
		require_once( ADINA_ADDONS . '/widgets/pricing.php' );
		require_once( ADINA_ADDONS . '/widgets/iconbox-two.php' );
		require_once( ADINA_ADDONS . '/widgets/testimonial-slider-section-two.php' );
		require_once( ADINA_ADDONS . '/widgets/about.php' );
		require_once( ADINA_ADDONS . '/widgets/hero-slider.php' );
		require_once( ADINA_ADDONS . '/widgets/tab.php' );
		require_once( ADINA_ADDONS . '/widgets/iconbox-three.php' );
		require_once( ADINA_ADDONS . '/widgets/accordion-section.php' );
		require_once( ADINA_ADDONS . '/widgets/cta.php' );
		require_once( ADINA_ADDONS . '/widgets/svg-icon.php' );
		require_once( ADINA_ADDONS . '/widgets/map.php' );
		require_once( ADINA_ADDONS . '/widgets/image-slider.php' );
		require_once( ADINA_ADDONS . '/widgets/history-section.php' );
		require_once( ADINA_ADDONS . '/widgets/service-list.php' );
		require_once( ADINA_ADDONS . '/widgets/document-list.php' );
		require_once( ADINA_ADDONS . '/widgets/accordion-two.php' );
		require_once( ADINA_ADDONS . '/widgets/tab-two.php' );
		
	}

    public function widget_scripts() {
        wp_enqueue_script(
            'adina-frontend-script',
            ADINA_PLUGDIRURI . 'assets/js/adina-frontend.js',
            array('jquery'),
            false,
            true
		);
		wp_enqueue_style(
            'adina-widget-style',
            ADINA_PLUGDIRURI . 'assets/css/style.css',
			microtime()
		);
	}

	public function editor_scripts() {
		wp_enqueue_script(
			'adina-core-editor',
			ADINA_PLUGDIRURI . 'assets/js/editor.js',
			array('jquery'),
			microtime(),
			true
		);
	}

    function adina_elementor_widget_categories( $elements_manager ) {
        $elements_manager->add_category(
            'adina',
            [
                'title' => __( 'Adina', 'adina' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
        $elements_manager->add_category(
            'adina_footer_elements',
            [
                'title' => __( 'Adina Footer Elements', 'adina' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);
		$elements_manager->add_category(
            'adina_header_elements',
            [
                'title' => __( 'Adina Header Elements', 'adina' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

	}

}
Adina_Extension::instance();